/* Copyright (c) 2016-2018 VMware, Inc. All rights reserved. */
package com.vmware.automatic.plugin.registration.services;

import com.vmware.vim25.InvalidLocaleFaultMsg;
import com.vmware.vim25.InvalidLoginFaultMsg;
import com.vmware.vim25.ManagedObjectReference;
import com.vmware.vim25.RuntimeFaultFaultMsg;
import com.vmware.vim25.ServiceContent;
import com.vmware.vim25.VimPortType;
import com.vmware.vim25.VimService;
import org.apache.commons.cli.CommandLine;

import javax.xml.ws.BindingProvider;
import java.util.Map;

/**
 * Service that provides public methods to connect/disconnect to/from
 * a vCenter Server.
 */
public class ConnectionService {

   /**
    * Service instance ref
    */
   private static final ManagedObjectReference SVC_INST_REF = new ManagedObjectReference();

   /**
    * Name and Type of Service instance
    */
   private static final String SVC_INST_NAME = "ServiceInstance";
   private final VimService vimService;
   private final TrustAllTrustManager trustManager;

   private VimPortType vimPort;
   private ServiceContent serviceContent;
   private ManagedObjectReference extensionManager;
   private boolean isConnected;

   public ConnectionService(VimService vimService,
         TrustAllTrustManager trustManager) {
      this.vimService = vimService;
      this.trustManager = trustManager;
   }

   /**
    * Creates session with the vCenter Server
    *
    * @param commandLine - command line arguments
    */
   public void connect(CommandLine commandLine) {
      if (isConnected) {
         return;
      }
      final String _username = commandLine.getOptionValue("u");
      final String _password = commandLine.getOptionValue("p");
      final String url = commandLine.getOptionValue("url");
      try {
         trustManager.trustAll();

         SVC_INST_REF.setType(SVC_INST_NAME);
         SVC_INST_REF.setValue(SVC_INST_NAME);

         vimPort = vimService.getVimPort();
         Map<String, Object> ctxt = ((BindingProvider) vimPort)
               .getRequestContext();

         ctxt.put(BindingProvider.ENDPOINT_ADDRESS_PROPERTY, url);
         ctxt.put(BindingProvider.SESSION_MAINTAIN_PROPERTY, true);

         serviceContent = vimPort.retrieveServiceContent(SVC_INST_REF);
         vimPort.login(serviceContent.getSessionManager(), _username, _password,
               null);

         extensionManager = serviceContent.getExtensionManager();
         isConnected = true;
      } catch (RuntimeFaultFaultMsg | InvalidLocaleFaultMsg | InvalidLoginFaultMsg e) {
         final String msg = String.format(
               "Error while connecting to vCenter Server SDK <%s>. Please verify the provided arguments are correct: -url: %s, -username: %s, -password: %s",
               url, url, _username, _password);
         throw new RuntimeException(msg, e);
      }
   }

   /**
    * Destroys session to vCenter Server.
    *
    * @throws RuntimeFaultFaultMsg
    */
   public void disconnect(CommandLine commandLine) {
      if (!isConnected) {
         return;
      }
      final String url = commandLine.getOptionValue("url");
      try {
         getVimPort().logout(serviceContent.getSessionManager());
      } catch (RuntimeFaultFaultMsg e) {
         final String msg = String.format(
               "Error while disconnecting from vCenter Server SDK <%s>.", url);
         throw new RuntimeException(msg, e);
      } finally {
         isConnected = false;
      }
   }

   /**
    * Get VimPortType instance
    */
   public VimPortType getVimPort() {
      if (vimPort == null) {
         throw new RuntimeException(
               "No Connection to vCenter Server established");
      }
      return vimPort;
   }

   /**
    * Get extension manager instance.
    */
   public ManagedObjectReference getExtensionManager() {
      if (extensionManager == null) {
         throw new RuntimeException(
               "No connection to vCenter Server established.");
      }
      return extensionManager;
   }
}
